# Contributing to stdarch

The `stdarch` crate is more than willing to accept contributions! First you'll
probably want to check out the repository and make sure that tests pass for you:

```
$ git clone https://github.com/rust-lang/stdarch
$ cd stdarch
$ TARGET="<your-target-arch>" ci/run.sh
```

Where `<your-target-arch>` is the target triple as used by `rustup`, e.g. `x86_64-unknown-linux-gnu` (without any preceding `nightly-` or similar).
Also remember that this repository requires the nightly channel of Rust!
The above tests do in fact require nightly rust to be the default on your system, to set that use `rustup default nightly` (and `rustup default stable` to revert).

If any of the above steps don't work, [please let us know][new]!

Next up you can [find an issue][issues] to help out on, we've selected a few
with the [`help wanted`][help] and [`impl-period`][impl] tags which could
particularly use some help. You may be most interested in [#40][vendor],
implementing all vendor intrinsics on x86. That issue's got some good pointers
about where to get started!

If you've got general questions feel free to [join us on gitter][gitter] and ask
around! Feel free to ping either @BurntSushi or @alexcrichton with questions.

[gitter]: https://gitter.im/rust-impl-period/WG-libs-simd

# How to write examples for stdarch intrinsics

There are a few features that must be enabled for the given intrinsic to work
properly and the example must only be run by `cargo test --doc` when the feature
is supported by the CPU. As a result, the default `fn main` that is generated by
`rustdoc` will not work (in most cases). Consider using the following as a guide
to ensure your example works as expected.

```rust
/// # // We need cfg_target_feature to ensure the example is only
/// # // run by `cargo test --doc` when the CPU supports the feature
/// # #![feature(cfg_target_feature)]
/// # // We need target_feature for the intrinsic to work
/// # #![feature(target_feature)]
/// #
/// # // rustdoc by default uses `extern crate stdarch`, but we need the
/// # // `#[macro_use]`
/// # #[macro_use] extern crate stdarch;
/// #
/// # // The real main function
/// # fn main() {
/// #     // Only run this if `<target feature>` is supported
/// #     if cfg_feature_enabled!("<target feature>") {
/// #         // Create a `worker` function that will only be run if the target feature
/// #         // is supported and ensure that `target_feature` is enabled for your worker
/// #         // function
/// #         #[target_feature(enable = "<target feature>")]
/// #         unsafe fn worker() {
///
/// // Write your example here. Feature specific intrinsics will work here! Go wild!
///
/// #         }
/// #         unsafe { worker(); }
/// #     }
/// # }
```

If some of the above syntax does not look familiar, the [Documentation as tests] section
of the [Rust Book] describes the `rustdoc` syntax quite well. As always, feel free
to [join us on gitter][gitter] and ask us if you hit any snags, and thank you for helping
to improve the documentation of `stdarch`!

# Alternative Testing Instructions

It is generally recommended that you use `ci/run-docker.sh` to run the tests.
However this might not work for you, e.g. if you are on Windows.

In that case you can fall back to running `cargo +nightly test` and `cargo +nightly test --release -p core_arch` for testing the code generation.
Note that these require the nightly toolchain to be installed and for `rustc` to know about your target triple and its CPU.
In particular you need to set the `TARGET` environment variable as you would for `ci/run.sh`.
In addition you need to set `RUSTCFLAGS` (need the `C`) to indicate target features, e.g. `RUSTCFLAGS="-C -target-features=+avx2"`.
You can also set `-C -target-cpu=native` if you're "just" developing against your current CPU.

Be warned that when you use these alternative instructions, [things may go less smoothly than they would with `ci/run-docker.sh`][ci-run-good], e.g. instruction generation tests may fail because the disassembler named them differently, e.g. it may generate `vaesenc` instead of `aesenc` instructions despite them behaving the same.
Also these instructions execute less tests than would normally be done, so don't be surprised that when you eventually pull-request some errors may show up for tests not covered here.


[new]: https://github.com/rust-lang/stdarch/issues/new
[issues]: https://github.com/rust-lang/stdarch/issues
[help]: https://github.com/rust-lang/stdarch/issues?q=is%3Aissue+is%3Aopen+label%3A%22help+wanted%22
[impl]: https://github.com/rust-lang/stdarch/issues?q=is%3Aissue+is%3Aopen+label%3Aimpl-period
[vendor]: https://github.com/rust-lang/stdarch/issues/40
[Documentation as tests]: https://doc.rust-lang.org/book/first-edition/documentation.html#documentation-as-tests
[Rust Book]: https://doc.rust-lang.org/book/first-edition
[ci-run-good]: https://github.com/rust-lang/stdarch/issues/931#issuecomment-711412126
